/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/* SCR; 28.10.97
 * Abstract:
 * This file contains I/O routines for the C-API.  These routines are
 *
 * DRIVER(PORT) and OS INDEPENDENT,
 * BEST-CARD DEPENDENT.
 *
 * The main purposes of the functions in this file:
 * 1. to encapsulate the algorithms (only) so that all "ports" are handled
 *    in the same manner.
 * 2. to reduce the amount of code necessary for each port driver.
 *
 * Some assumptions are made about any port;
 * 1. Each port has a driver such that BestBasicRead/Write can be called.
 * 2. The port has been "opened" (i.e. BestOpen() called) before any routines
 *    here are called.
 * 3. Each port should provide the following primitive routines or use the gen.
 *    purpose ones here.  (substitute a 3-letter port abbreviation for ***);
 *
 *    Best***RegwidthSet()
 *    Best***DeviceConnect()
 *    Best***CheckConnection()
 *    Best***ReleaseConnection()
 *
 *    Not normally needed but available...
 *    Best***OnReadError()
 *    Best***OnWriteError()
 *
 * See additional notes in b_io.h
 */

#if defined(_MSC_VER)
# pragma data_seg("iocommon_c","FAR_DATA")
# pragma warning(disable: 4705) /* params not used (statement has no effect) */
#elif defined(__BORLANDC__)
# pragma option -dc
# pragma warn -eff
#endif /* _MSC_VER etc. */


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include <typedefs.h>

#include <iocommon.h>

#include "b_io.h"
#include "b_cmd.h"
#include "pci.h"
#include "hif.h"
#include "b_serial.h"
#include "session.h"
#include "lasterr.h"

#include <errcapi.h>
#include <regconst.h>
#include <regx10.h>
#include <regx11.h>

#include "timeout.h"


/* for debug tracing */
#undef BBC_PROGRESS

#if defined (BEST_DEBUG) && defined (BBC_PROGRESS)
# define DBG_CB_PRINTF(msg) { \
    if (handle_array[handle].cb_printf) \
      handle_array[handle].cb_printf((msg));}
#else
# define DBG_CB_PRINTF(msg)
#endif

#undef BBBC_PROGRESS

#if defined (BEST_DEBUG) && defined (BBBC_PROGRESS)
# define DBG_CB_PRINTF(msg) { \
    if (handle_array[handle].cb_printf) \
      handle_array[handle].cb_printf((msg));}
#else
# define DBG_CB_PRINTF(msg)
#endif


#define EIGHT_LA "<", "<", "<", "<", "<", "<", "<", "<"
#define EIGHT_RA ">", ">", ">", ">", ">", ">", ">", ">"

static char    *cb_read_string_array [MAXHANDLES] = {
  		  EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA,
  		  EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA,
  		  EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA,
  		  EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA
		};

static char    *cb_write_string_array [MAXHANDLES] = {
  		  EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA,
  		  EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA,
  		  EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA,
  		  EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA
		};

#if MAXHANDLES > 128
#error Initialization of cb_xx_string_array invalidated by increase in MAXHANDLES
#endif


extern b_int32	best_capi_prop [MAXHANDLES + 1][B_CAPIPROP_SIZE];
/* --------------------------------------------------------------------------
 * STATIC FUNCTIONS
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * Some low-level drivers have a bus-width > 1 byte and thus can do their
 * own byte-ordering (see Notes on RegWidth above).
 * Thus the name of this function could also be
 * BestIsPortBusWidthMoreThanEightBits() (but that's too long).
 * -------------------------------------------------------------------------- */

static int BestIsIoByteOrdered(b_handletype handle)
{
  switch (handle_array[handle].port)
  {
    case B_PORT_OFFLINE:
    case B_PORT_RS232:
    case B_PORT_PARALLEL:
      break;

    case B_PORT_PCI_CONF:
    case B_PORT_PCI_IO:
      return 1;

    case B_PORT_FASTHIF:
      break;

    default:
      assert(0 && "Illegal port in call to BestIsIoByteOrdered()"); /*lint !e506 */
      return 0;  /*lint !e527 .. unreachable */
  }	/*lint !e788 ... not all enums used */
  return 0;
}


/* --------------------------------------------------------------------------
 * Each port type MUST have an appropriate Best***DeviceConnect().
 * -------------------------------------------------------------------------- */

static b_errtype BestDeviceConnect(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;

  /* handle was already checked before */
  b_portnumtype OsHandle = handle_array[handle].portnumber;

  B_TRY_BEGIN {
    switch (handle_array[handle].port) {
     case B_PORT_OFFLINE:
      handle_array[handle].is_connected = 1;
      break;

     case B_PORT_RS232:
      B_TRY (BestSerDeviceConnect(OsHandle));
      break;

     case B_PORT_PCI_CONF:
     case B_PORT_PCI_IO:
      B_TRY (BestPCIDeviceConnect(OsHandle));
      break;

     case B_PORT_FASTHIF:
      /* ignore return value, CZ */
      (void)BestHIFDeviceConnect (OsHandle);
      break;

     default:
      return B_E_WRONG_PORT;
    }	/*lint !e788 ... not all enums used */
  }

  /* TODO: check out whether the catch can be eliminated */
  B_TRY_CATCH {
    B_TRY_RET = B_E_CANNOT_CONNECT;
    BestLastErrorParamSet (handle, B_ERRPAR_1, (b_int32)handle_array[handle].port);
  }
  
  return B_TRY_RET;
}


/* --------------------------------------------------------------------------
 * Some ports need an ack from the Best
 * -------------------------------------------------------------------------- */

static int BestBasicNeedsAck(b_handletype handle)
{
  switch (handle_array[handle].port)
  {
    case B_PORT_OFFLINE:
      break;

    case B_PORT_RS232:
      return 1;

    case B_PORT_PARALLEL:
    case B_PORT_PCI_CONF:
    case B_PORT_PCI_IO:
    case B_PORT_FASTHIF:
      break;

    default:
      assert(0 && "Illegal port in call to BestBasicNeedsAck()"); /*lint !e506 */
      return 0; /*lint !e527 .. unreachable */
  }	/*lint !e788 ... not all enums used */
  return 0;
}


/* --------------------------------------------------------------------------
 * returns B_E_OK if port closed during the timeout period (in ms.)
 * -------------------------------------------------------------------------- */

b_errtype BestWaitForClose(b_handletype handle, b_int32 timeout)
{
  b_errtype err;
  BEST_TIMER tmrTotal;
  BestStartTimeout(timeout, &tmrTotal);

  while (B_E_OK == (err = BestIsDisconnected(handle)))
  {
    if (BestIsTimeoutDone(&tmrTotal)) {
      /* TODO: better error message ? */
      BestLastErrorParamSet (handle, B_ERRPAR_1, (b_int32)B_EERR_CAPI_TIMEOUT);
      return B_E_ERROR;
    }
  }

  if (B_E_NOT_CONNECTED == err)
    return B_E_OK;

  return err;
}


/* --------------------------------------------------------------------------
 * return the maximum number of bytes able to be input or output to/from
 * any port in one read or write.
 * TODO; make real
 * -------------------------------------------------------------------------- */

#if 0
/* TODO: obsolete function, take out, CZ */
static b_int32 BestGetMaxBytesPerIo(b_handletype handle)
{
  switch (handle_array[handle].port)
  {
   case B_PORT_OFFLINE:
   case B_PORT_RS232:
   case B_PORT_PARALLEL:
   case B_PORT_PCI_CONF:
   case B_PORT_PCI_IO:
   case B_PORT_FASTHIF:
    break;

   default:
    assert (0);
    break;
  }

  if (Best16BitRegisterFile(handle))
  {
    return MAX_BLOCK_LEN_2926;
  }
  else
  {
    return MAX_BLOCK_LEN_2925;
  }
}
#endif


/* --------------------------------------------------------------------------
 * return the maximum date width (in bytes) of any port
 * TODO; make real
 * -------------------------------------------------------------------------- */

static b_int8 BestGetBlockRegWidth(b_handletype handle)
{
  switch (handle_array[handle].port)
  {
  case B_PORT_OFFLINE:
  case B_PORT_RS232:
  case B_PORT_PARALLEL:
    break;

  case B_PORT_PCI_CONF:
  case B_PORT_PCI_IO:
    return 4;

  case B_PORT_FASTHIF:
    break;

  default:
   assert (0);  /*lint !e506 */
  }	/*lint !e788 ... not all enums used */

  return 1;                     /* 64K default */
}


/* --------------------------------------------------------------------------
 * 2925 and 2926 w. reg# < 0x0a only
 * -------------------------------------------------------------------------- */

/* TODO: should not be exported */

b_errtype EXPORT BestBasicBlockWrite(b_handletype handle,
    b_int32 regnum,
    b_int8ptr bt_ptr,
    b_int64 num_of_bytes)
{
  if (Best16BitRegisterFile(handle) &&
    (regnum > REG_COMMON_MAX) && !IN_OFFLINE_MODE)
  {
    BestLastErrorParamSet(handle, B_ERRPAR_1, regnum);
    B_ERRETURN(B_E_UNTRANSLATED_CMD);
  }
  else
  {
    b_errtype err;
    b_sizetype regsize;
    /* scr; sorry about this but the options were limited */
    regsize = b_regsize[(size_t) regnum].CardModel[REG_INDEX];

    /* old protocol */
    err = BestBasicCommand8Bit(handle, regnum, CMD_WRITE,
      bt_ptr, regsize, num_of_bytes);

    /* TODO: why should I disconnect? */
    if (err == B_E_ERROR)
      (void) BestDisconnect(handle);

    B_ERRETURN(err);
  }
}



/* --------------------------------------------------------------------------
 * 2925 and 2926 w. reg# < 0x0a only
 * -------------------------------------------------------------------------- */

/* TODO: should not be exported */
b_errtype EXPORT BestBasicBlockRead(b_handletype handle,
    b_int32 regnum,
    b_int8ptr bt_ptr,
    b_int32 num_of_bytes)
{
  if (Best16BitRegisterFile(handle) &&
    (regnum > REG_COMMON_MAX) && !IN_OFFLINE_MODE)
  {
    BestLastErrorParamSet(handle, B_ERRPAR_1, regnum);
    B_ERRETURN(B_E_UNTRANSLATED_CMD);
  }

  else
  {
    b_errtype err;
    b_sizetype regsize;
    /* scr; sorry about this but the options were limited */
    regsize = b_regsize[(size_t) regnum].CardModel[REG_INDEX];

    err = BestBasicCommand8Bit(handle, regnum, CMD_READ,
      bt_ptr, regsize, num_of_bytes);

    if (err == B_E_ERROR)
      (void) BestDisconnect(handle);

    B_ERRETURN(err);
  }
}


/* BestBasicCommand
   The procedure consists of the following steps:
     1. send command code
     2. send parameters
     3. receive result
     4. receive return code
   Each of these steps increments the progress indicator of the try mechanism.
   If step 2 or 3 is skipped there is a dummy progress increment. The catch
   handler then computes the error message according to the achieved progress.
   CZ */
b_errtype EXPORT BestBasicCommand(b_handletype handle,
    b_int16 cmdcode,
    b_int8ptr inptr,
    b_int16 insize,
    b_int8ptr outptr,
    b_int16 * outsize)
{
  B_TRY_VARS;
  b_int8 buf[2] = {0, 0};
  b_int8 reply = 0;
  /* outsize may be null pointer only if outptr is null pointer */
  assert(outsize || !outptr);

  if (!insize)
    inptr = NULL;

  if (outsize)
  {
    if (!*outsize)
      outptr = NULL;
  }

  B_TRY_BEGIN
  {
    B_TRY_HANDLE;

    /* split command code in two bytes, second byte in first position */
    buf[1] = (b_int8) ((cmdcode & 0xff00) >> 8);
    buf[0] = (b_int8) (cmdcode & 0x00ff);

    /* send command code */
    B_TRY_PROGRESS(BestBasicByteStreamWrite(handle, buf, 2UL));
    DBG_CB_PRINTF(("BBC: opcode\n"));

    /* The card will disconnect if it cannot understand the command sent. */
    B_TRY(BestCheckConnection(handle));

    if (inptr)
    {
      if (insize == B_LENGTH_NEGOTIATE)
      {
        /* length is transmitted, first two bytes of buffer */
        B_TRY(BestBasicByteStreamWrite(handle, inptr, 2UL));
        inptr = BestStream2Word(&insize, inptr, 1UL);
      }

      /* transmit data */
      B_TRY_PROGRESS(BestBasicByteStreamWrite(handle, inptr, (b_int32) insize));
      DBG_CB_PRINTF(("BBC: parameters\n"));
    }

    else
    {
      /* synchronize progress */
      B_TRY_PROGRESS(B_E_OK);
    }

    if (outptr)
    {
      assert(outsize);

      /* read ONLY if you have valid pointers */
      if (*outsize == B_LENGTH_NEGOTIATE)
      {
        /* length is received from card */
        B_TRY(BestBasicByteStreamRead(handle, (b_int8ptr) outsize, 2UL));
        (void) BestStream2Word(outsize, (b_int8ptr) outsize, 1UL);
      }

      /* receive data */
      B_TRY_PROGRESS(BestBasicByteStreamRead(handle, outptr, (b_int32) * outsize));
      DBG_CB_PRINTF(("BBC: result\n"));
    }

    else
    {
      /* synchronize progress */
      B_TRY_PROGRESS(B_E_OK);
    }

    /* receive status */
    B_TRY_PROGRESS(BestBasicByteRead(handle, &reply));
    DBG_CB_PRINTF(("BBC: reply (%02x\\h)\n", (long) reply));

    B_TRY_FAIL(reply ? (b_errtype) reply + B_E_FIRMWARE : B_E_OK);

  }

  B_TRY_CATCH
  {
    if (cmdcode == CMD_ERRPAR_GET)
    {
      /* TODO: find good error message */
      return B_E_FUNC;
    }

    /* do not remove even empty try passed blocks, CZ */
    B_TRY_PASSED
    {
      /* command bytes */

      if (B_TRY_RET == B_E_NOT_CONNECTED)
      {
        B_TRY_RET = B_E_CONNECTION_LOST_CMD;
        BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32) buf[1]);
        BestLastErrorParamSet(handle, B_ERRPAR_3, (b_int32) buf[0]);
      }
    }

    B_TRY_PASSED
    {
      /* in data */

      if (B_TRY_RET == B_E_NOT_CONNECTED)
      {
        B_TRY_RET = B_E_CONNECTION_LOST;
      }
    }

    B_TRY_PASSED
    {
      /* out data */
    }

    B_TRY_PASSED
    {
      b_errtype err = B_E_FUNC;
      b_int8 outbuf[OUT_ERRPAR_GET];
      b_int32 errpar[B_ERROR_NUMPARAMS];
      b_int16 outputsize = OUT_ERRPAR_GET;
      /* return code */
      if (B_TRY_RET == B_EFW_LONG_OPERATION)
      {
        reply = (b_int8) (B_EFW_LONG_OPERATION - B_E_FIRMWARE);

        while (reply == (b_int8) (B_EFW_LONG_OPERATION - B_E_FIRMWARE))
        {
          err = BestBasicByteRead(handle, &reply);

          if (handle_array[handle].cb_printf != NULL)
            handle_array[handle].cb_printf(".");

          if (err)
          {
            /* error in transmission */
            return err;
          }
        }

        B_TRY_RET = (b_errtype) reply;
      }

      if (B_TRY_RET)
      {
        DBG_CB_PRINTF(("\n"));

        err = BestBasicCommand(handle, CMD_ERRPAR_GET,
          NULL, IN_ERRPAR_GET, outbuf, &outputsize);

        if (!err)
        {
          (void) BestStream2Long(errpar, outbuf, (b_int32) B_ERROR_NUMPARAMS);
          BestLastErrorAllParamsSet(handle,
            errpar[0], errpar[1], errpar[2],
            errpar[3], errpar[4]);
          B_TRY_RET = (b_errtype) reply + B_E_FIRMWARE;
        }
      }
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/******************************************************************************/
/* organizes transfers of variable lenght */
b_errtype EXPORT BestBasicCommandVariable(b_handletype handle,
    b_int16 cmdcode,
    b_int8ptr inptr,
    b_int16 insize,
    b_int8ptr outptr,
    b_int16 * outsize)
{
  b_int8ptr buf;
  b_errtype err;
  if ((buf = (b_int8ptr) malloc(insize + 2)) == NULL) /* create a new buffer */
  {
    return B_E_HOST_MEM_FULL;
  }

  if (inptr)
    memcpy(buf + 2, inptr, insize); /* copy from old to new buffer */

  /* put into buffer */
  (void) BestWord2Stream(buf, &insize, 1UL);

  err = BestBasicCommand(handle, cmdcode, buf,  /* call standard fct */
    B_LENGTH_NEGOTIATE, outptr, outsize);

  free(buf);                    /* free the allocated memory */

  return err;
}


/* --------------------------------------------------------------------------
 * BestBasicCommand8Bit() is designed to be used with any port.
 * As the name suggests it is restricted to the 2925 ONLY !!!
 * It is NOT intended for export!
 * N.B. This is currently setup to transfer a maximum of MAX_BLOCK_LEN_2925
 *      bytes per BestBasicWrite().
 * -------------------------------------------------------------------------- */

b_errtype BestBasicCommand8Bit(
    b_handletype handle,
    b_int32 addr,               /* 8-bit address to be sent to Best */
    b_int8 cmd,                 /* CMD_READ or CMD_WRITE */
    b_int8ptr  byteptr,           /* addr of data to be read/written */
    b_int8 regwidth,            /* using register width nn */
    b_int64 NumBytes)           /* length of data at byteptr */
{
  B_DECLARE_FUNCNAME ("BestBasicCommand8Bit");
  B_TRY_VARS_NO_PROG;
  b_int8 OneByteBuff;
  b_int8 NumDataEltPerIo;
  b_int64 NumDataEltLeft;
  char *cb_string = NULL;
  B_TRY_BEGIN
  {
    /* Note that we do run-time null ptr. checking here...below this level
     * we'll only be asserting */
    B_TRY_FAIL(NumBytes && (NULL == byteptr) ?
      B_E_PARAM_NULL_POINTER : B_E_OK);

    assert(BestIsRegWidthValid(regwidth)
	&& "Invalid reg. width in BestBasicCommand8Bit()");  /*lint !e506 */
    assert(0 == (NumBytes % regwidth)
    	&& "Boundary violation in BestBasicCommand8Bit()");  /*lint !e506 */

    NumDataEltLeft = NumBytes / (b_int64) regwidth;

    while (NumDataEltLeft > 0L)
    {
      if (NumDataEltLeft > MAX_BLOCK_LEN_2925)
      {
        NumDataEltPerIo = MAX_BLOCK_LEN_2925;
        NumDataEltLeft -= (b_int32) NumDataEltPerIo;
      }

      else
      {
        NumDataEltPerIo = (b_int8) NumDataEltLeft;
        NumDataEltLeft = 0L;
      }

      /* connection is checked before each cmd */
      B_TRY(BestCheckConnection(handle));

      /* send preamble          */
      /* byte 0: cmd            */
      /* byte 1: reg-no         */
      /* byte 2: NumBytes (accesses) */

      OneByteBuff = (b_int8) ((NumDataEltPerIo & '\x7f') | (cmd << 7));
      B_TRY(BestBasicByteWrite(handle, &OneByteBuff));

      /* A 2925 on the serial port will ack HERE. */
      if (BestIs2925 (handle) && BestBasicNeedsAck(handle))
      {
        B_TRY(BestBasicByteRead(handle, &OneByteBuff));
      }

      OneByteBuff = (b_int8) addr;
      B_TRY(BestBasicByteWrite(handle, &OneByteBuff));

      /* All EXCEPT a 2925 on the serial port will ack HERE. */
      if (!BestIs2925 (handle) && BestBasicNeedsAck(handle))
      {
        B_TRY(BestBasicByteRead(handle, &OneByteBuff));
      }

      OneByteBuff = STATUS_OK;

      switch (cmd)
      {
      case CMD_READ:
        /* get data from Mini */
        B_TRY(BestBasicByteOrderRead(handle,
            byteptr,
	    (b_int32)NumDataEltPerIo,
	    regwidth));

	byteptr += (int) (NumDataEltPerIo * regwidth);
        cb_string = cb_read_string_array [handle];
        break;

      case CMD_WRITE:
        /* send data to Mini */
        B_TRY(BestBasicByteOrderWrite(handle,
            byteptr,
	    (b_int32)NumDataEltPerIo,
	    regwidth));

        byteptr += (int) (NumDataEltPerIo * regwidth);
        cb_string = cb_write_string_array [handle];
        break;

      default:
        B_TRY_FCT_PARAM (cmd, (cmd != CMD_WRITE) && (cmd != CMD_READ));
      }

      /* get status from Mini */
      B_TRY(BestBasicByteRead(handle, &OneByteBuff));

      /* call back */
      if (handle_array[handle].cb_printf != NULL)
        handle_array[handle].cb_printf(cb_string);

      if (!BestIs2925(handle))
      {
	B_TRY_FAIL(OneByteBuff ? OneByteBuff + B_E_FIRMWARE : B_E_OK);
      }
      else
      {
        switch (OneByteBuff)
        {
	 case STATUS_OK:
          break;

	 case STATUS_FUNC_ERROR:
	  B_TRY_ERROR(B_E_FUNC);

	 default:
	  BestLastErrorParamSet (handle, B_ERRPAR_1, (b_int32)B_EERR_UNKNOWN_REPLY);
	  B_TRY_ERROR(B_E_ERROR);
        }
      }

      /* next block */
    }
  }

  B_TRY_CATCH
  {
    if (B_TRY_RET == B_EFW_LONG_OPERATION)
    {
      OneByteBuff = B_EFW_LONG_OPERATION - B_E_FIRMWARE;

      while (OneByteBuff == B_EFW_LONG_OPERATION - B_E_FIRMWARE)
      {
        b_errtype err = B_E_FUNC;
        err = BestBasicByteRead(handle, &OneByteBuff);

        if (handle_array[handle].cb_printf != NULL)
          handle_array[handle].cb_printf(".");

        if (err)
        {
          /* error in transmission */
          return err;
        }
      }

      B_TRY_RET = OneByteBuff;
    }
  }

  return B_TRY_RET;
}


/* --------------------------------------------------------------------------
 * The routine to be called for ANY port to open a connection...timeout in ms
 * -------------------------------------------------------------------------- */

b_errtype BestOpenConnection(b_handletype handle, b_int32 timeout)
{
  B_TRY_VARS_NO_PROG;
  BEST_TIMER tmrTotal;

  B_TRY_BEGIN {
    /* try to connect */
    B_TRY (BestDeviceConnect(handle));
    /* start timeout */
    BestStartTimeout(timeout, &tmrTotal);

    /* check the validity of the connection. */
    /* if there's no timeout we'll loop forever (or until there's an error) */
    while (BestCheckConnection(handle) != B_E_OK)
    {
      if (BestIsTimeoutDone(&tmrTotal)) {
	BestLastErrorParamSet (handle, B_ERRPAR_1,
			       (b_int32)handle_array[handle].port);
	B_TRY_ERROR (B_E_CANNOT_CONNECT);
      }
    }
  }
  
  return B_TRY_RET;
}


/* --------------------------------------------------------------------------
 * See notes at the beginning of this file
 * -------------------------------------------------------------------------- */

b_errtype BestRegwidthSet(b_handletype handle, b_int8 RegWidth)
{
  b_portnumtype OsHandle;
  assert(BestIsRegWidthValid(RegWidth));

  /* get out if there's no change from last time */
  if (RegWidth == handle_array[handle].regwidth)
    return B_E_OK;

  /* set new value */
  handle_array[handle].regwidth = RegWidth;

  OsHandle = handle_array[handle].portnumber;

  switch (handle_array[handle].port)
  {
    case B_PORT_OFFLINE:
    case B_PORT_RS232:
    case B_PORT_PARALLEL:         /* SCR; 27.11.97; all par. drivers byte-aware
                                   * only */
      break;

    case B_PORT_PCI_CONF:
    case B_PORT_PCI_IO:
      return BestPCISetRegwidth(OsHandle, RegWidth);

    case B_PORT_FASTHIF:
      /* SCR; 20.11.97; driver is byte-aware only */
      break;

    default:
      return B_E_WRONG_PORT;
  }  /*lint !e788 */

  assert((RegWidth == 1)
	&& "Only byte-streams call default BestRegwidthSet()");  /*lint !e506 */

  return B_E_OK;
}


/* --------------------------------------------------------------------------
 * 2 different means of "checking" the connection...
 * a. Hardware handshake (DTR/DSR etc.)
 * b. PCI Config. Register
 * -------------------------------------------------------------------------- */

b_errtype BestCheckConnection(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  
  b_portnumtype OsHandle = handle_array[handle].portnumber;

  B_TRY_BEGIN {
    switch (handle_array[handle].port) {
     case B_PORT_OFFLINE:
      B_TRY_FAIL (BestIsHandleConnected(handle) ? B_E_OK : B_E_NOT_CONNECTED);
      break;
      
     case B_PORT_RS232:
      B_TRY (BestSerCheckConnection(OsHandle));
      break;
      
     case B_PORT_PCI_CONF:
     case B_PORT_PCI_IO:
      B_TRY (BestPCICheckConnection(OsHandle));
      break;
      
     case B_PORT_FASTHIF:
      B_TRY (BestHIFCheckConnection(OsHandle));
      break;

     default:
      return B_E_PARAM;
    }  /*lint !e788 */
  }

  B_TRY_CATCH {
    B_TRY_RET = B_E_NOT_CONNECTED;
    BestLastErrorParamSet (handle, B_ERRPAR_1, (b_int32)handle_array[handle].port);
  }
  
  return B_TRY_RET;
}


b_errtype BestIsDisconnected (b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  
  b_portnumtype OsHandle = handle_array[handle].portnumber;

  B_TRY_BEGIN {
    switch (handle_array[handle].port) {
     case B_PORT_OFFLINE:
      B_TRY_FAIL (BestIsHandleConnected(handle) ? B_E_OK : B_E_NOT_CONNECTED);
      break;
      
     case B_PORT_RS232:
      B_TRY (BestSerCheckConnection(OsHandle));
      break;
      
     case B_PORT_PCI_CONF:
     case B_PORT_PCI_IO:
      B_TRY (BestPCICheckConnection(OsHandle));
      break;
      
     case B_PORT_FASTHIF:
      B_TRY (BestHIFIsDisconnected(OsHandle));
      break;

     default:
      return B_E_PARAM;
    }  /*lint !e788 */
  }

  B_TRY_CATCH {
    B_TRY_RET = B_E_NOT_CONNECTED;
    BestLastErrorParamSet (handle, B_ERRPAR_1, (b_int32)handle_array[handle].port);
  }
  
  return B_TRY_RET;
}


/* --------------------------------------------------------------------------
 *
 * -------------------------------------------------------------------------- */

void BestReleaseConnection(b_handletype handle)
{
  /* Disconnect is somehow not fully implemented for the E2925A
     under PCI connection.
     Do not wait for it. CZ */

  b_portnumtype OsHandle = handle_array[handle].portnumber;
  
  switch (handle_array[handle].port) {
   case B_PORT_OFFLINE:
    handle_array[handle].is_connected = 0;
    break;

   case B_PORT_RS232:
    BestSerReleaseConnection(OsHandle);
    break;

   case B_PORT_PCI_CONF:
   case B_PORT_PCI_IO:
    BestPCIReleaseConnection(OsHandle);
    break;

  case B_PORT_FASTHIF:
    BestHIFReleaseConnection(OsHandle);
    break;

  default:
    assert(0 && "Bad port in handle array; BestReleaseConnection()"); /*lint !e506 */
    break;
  }  /*lint !e788 */

  (void)BestWaitForClose(handle, 1000UL);

  return;
}


/* --------------------------------------------------------------------------
 * These functions allow special actions or error messages following a
 * failure of BestBasicRead() or BestBasicWrite().
 * They should not normally need to be implemented.  Serial comm. is one
 * exception.
 * -------------------------------------------------------------------------- */

b_errtype BestOnReadError(b_handletype handle)
{
  DBG_ApiLastError;

  switch (handle_array[handle].port)
  {
  case B_PORT_OFFLINE:
    break;

  case B_PORT_RS232:
    return BestSerOnReadOrWriteError(handle_array[handle].portnumber);

  case B_PORT_PARALLEL:
  case B_PORT_PCI_CONF:
  case B_PORT_PCI_IO:
  case B_PORT_FASTHIF:
    break;

  default:
   assert (0);  /*lint !e506 */
  }  /*lint !e788 */

  BestLastErrorParamSet (handle, B_ERRPAR_1, (b_int32)B_EERR_GENERIC_READ);
  return B_E_ERROR;
}

b_errtype BestOnWriteError(b_handletype handle)
{
  DBG_ApiLastError;

  switch (handle_array[handle].port)
  {
  case B_PORT_OFFLINE:
    break;

  case B_PORT_RS232:
    return BestSerOnReadOrWriteError(handle_array[handle].portnumber);

  case B_PORT_PARALLEL:
  case B_PORT_PCI_CONF:
  case B_PORT_PCI_IO:
  case B_PORT_FASTHIF:
    break;

  default:
   assert (0);  /*lint !e506 */
  }  /*lint !e788 */
  
  BestLastErrorParamSet (handle, B_ERRPAR_1, (b_int32)B_EERR_GENERIC_WRITE);
  return B_E_ERROR;
}


/* --------------------------------------------------------------------------
 * This function allows any driver to check for a user's callback function
 * during read/write.  It's called from BestBasicRead().
 * Returns an error code (...stop processing on error) or B_E_OK
 * -------------------------------------------------------------------------- */

b_errtype BestCheckCallBack(b_handletype handle)
{
  if (handle_array[handle].cb_getevent &&
    handle_array[handle].cb_getevent() != 0)
  {
    BestLastErrorParamSet (handle, B_ERRPAR_1, (b_int32)B_EERR_UNKNOWN);
    return B_E_ERROR;
  }

  return B_E_OK;
}


/* --------------------------------------------------------------------------
 * Determination of 'endianess'.  Note; This is the "real" answer.
 * To save time it can be defined true (i.e. Intel only) in b_io.h but
 * this function is pretty quick.
 * !!! THUS; DO NOT CALL THIS FUNCTION DIRECTLY !!!
 * -------------------------------------------------------------------------- */

static unsigned char chEndian[] = {0x12,0x34,0x56,0x78};

int BestIsCurMachineLittleEndian(void)
{
  return (*(unsigned long *)chEndian == 0x78563412UL);
}
/* --------------------------------------------------------------------------
 * This function will read a stream from the current port and change the
 * byte order as necessary (depending on datatype/regwidth).
 * See b_io.h for additional macros which use this function.
 * -------------------------------------------------------------------------- */

b_errtype BestBasicByteOrderRead(
    b_handletype handle,
    b_int8ptr  pData,             /* caller's buffer */
    b_int32 NumDataElt,         /* number of data elements to read (NOT
                                 * bytes) */
    b_int8 DataEltWidth)        /* data element width in bytes */
{
  b_errtype err;
  int fIsIoByteOrdered = BestIsIoByteOrdered(handle);
  /* if the low level driver can only handle streams then we'll read using a
   * "fake" data-element width ... but that doesn't change our byte-ordering
   * requirements later on !! */

  b_int8 StreamWidth = (b_int8)(fIsIoByteOrdered ? DataEltWidth : 1);
  if (NULL == pData)
    return B_E_PARAM_NULL_POINTER;

  /* first "dump" the number of bytes from the port into the user's buffer */
  if (B_E_OK != (err = BestBasicRead(handle,
        pData,
        NumDataElt * DataEltWidth,
        StreamWidth))
    )
    return err;

  /* do we need to change the byte order ? */
  if (fIsIoByteOrdered || (DataEltWidth == 1) || !BestIsHostLittleEndian())
    return B_E_OK;

  /* OH, DARN...we've got work to do. The cast to void * gets rid of
   * misalignment issues on different platforms */
  switch (DataEltWidth)
  {
  case 2:
    (void) BestStream2Word((b_int16 *) (void *) pData, pData, NumDataElt);
    break;

  case 4:
    (void) BestStream2Long((b_int32 *) (void *) pData, pData, NumDataElt);
    break;

  default:
    assert(0
	&& "Illegal data type (register width) in BestBasicByteOrderRead");  /*lint !e506 */
    break;
  }
  return B_E_OK;
}


#define BBBOW_BUF_SIZE  12

/* --------------------------------------------------------------------------
 * This function will write the caller's buffer "byte-ordered" to a port.
 * It malloc's buffers that are over BBBOW_BUF_SIZE bytes long
 * -------------------------------------------------------------------------- */

b_errtype BestBasicByteOrderWrite(
    b_handletype handle,
    b_int8ptr  pData,             /* caller's buffer */
    b_int32 NumDataElt,         /* number of data elements to write (NOT
                                 * bytes) */
    b_int8 DataEltWidth)        /* data element width in bytes */
{
  b_int8 BoBuff[BBBOW_BUF_SIZE];
  b_errtype err;
  b_int8 fMallocUsed = 0;
  b_int8ptr pOurBuff = pData;
  b_int32 NumBytes = (NumDataElt * DataEltWidth);
  if (NULL == pData)
    return B_E_PARAM_NULL_POINTER;

  /* do we need to change the order or will the driver do it? OR did we get
   * lucky and get a byte stream anyway? */
  if (!BestIsIoByteOrdered(handle) && (DataEltWidth > (b_int8) 1))
  {
    /* we only need to change byte order for little-endian hosts but we do
     * need to change the data-element width...regardless */
    if (BestIsHostLittleEndian())
    {
      /* do we need to malloc ? */
      if (NumBytes > sizeof(BoBuff))
      {
        if (NULL == (pOurBuff = (b_int8ptr ) malloc((size_t) NumBytes)))
          return B_E_HOST_MEM_FULL;
        fMallocUsed = 1;
      }
      else
      {
        pOurBuff = BoBuff;
      }

      /* do the conversion. The cast to void * gets rid of misalignment
       * issues on different platforms */
      switch (DataEltWidth)
      {
      case 2:
        (void) BestWord2Stream(pOurBuff, (b_int16 *) (void *) pData, NumDataElt);
        break;

      case 4:
        (void) BestLong2Stream(pOurBuff, (b_int32 *) (void *) pData, NumDataElt);
        break;

      default:
	assert(0
		&& "Illegal data type (register width) in BestBasicByteOrderRead");  /*lint !e506 */
        break;
      }
    }

    /* after conversion to a stream the data element width is 1 byte */
    DataEltWidth = 1;
  }

  err = BestBasicWrite(handle, pOurBuff, NumBytes, DataEltWidth);

  if (fMallocUsed)
  {
    free(pOurBuff);
  }
  return err;
}
/* Default timeouts used for each port.
 * Note that it is only necessary to set the TIMEOUTS_DEF_xxxx in the
 * header file.  No other action need be taken by the port.
 * This default value is set each time the port is opened.
 * Kernel mode drivers need to set their own defaults if they read/write
 * as part of an open (or read/write without opening...mailbox).
 */
static BESTTIMEOUTS DefaultSerTimeouts = TIMEOUTS_E2925_SERIAL;
static BESTTIMEOUTS DefaultParTimeouts = TIMEOUTS_DEF_PAR;
static BESTTIMEOUTS DefaultPciTimeouts = TIMEOUTS_DEF_PCI;
static BESTTIMEOUTS DefaultHifTimeouts = TIMEOUTS_DEF_FHIF;

static BESTTIMEOUTS NoTimeouts = NO_TIMEOUTS;
static BESTTIMEOUTS ImmReadTimeout = IMM_READ_TIMEOUT;
static BESTTIMEOUTS ImmWriteTimeout = IMM_WRITE_TIMEOUT;
static BESTTIMEOUTS ImmIoTimeouts = IMM_IO_TIMEOUTS;
/* --------------------------------------------------------------------------
 * Sets port timeouts.
 * note that we store the timeouts in the handle array
 * ... some multi-device drivers can't easily provide their own storage
 *
 * Passing
 *    pCallersTimeouts == NULL OR fSetAction = TIMEOUTS_SET_DEF
 *      will set the DEFAULT timeouts.
 *    fSetAction = TIMEOUTS_SET_NONE will set no timeouts (i.e. infinite).
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestPortTimeoutSet(
                        b_handletype handle,
                        BESTTIMEOUTS * pCallersTimeouts,
                        int fSetAction)
{
  /* our own pointer */
  BESTTIMEOUTS *pHandleTimeouts = &(handle_array[handle].timeouts);
  b_portnumtype OsHandle = handle_array[handle].portnumber;

  /* as the defaults are port-dependent we handle those differently */
  int fSetDefault = ((NULL == pCallersTimeouts) && (TIMEOUTS_SET_DEF == fSetAction));

  if (!fSetDefault)
  {
    switch (fSetAction)
    {
    case TIMEOUTS_SET:
      if(NULL == pCallersTimeouts)
        return B_E_PARAM_NULL_POINTER;
      *pHandleTimeouts = *pCallersTimeouts;
      break;

    case TIMEOUTS_SET_NONE:
      *pHandleTimeouts = NoTimeouts;
      break;

    case TIMEOUTS_SET_IMM_READ:
      *pHandleTimeouts = ImmReadTimeout;
      break;

    case TIMEOUTS_SET_IMM_WRITE:
      *pHandleTimeouts = ImmWriteTimeout;
      break;

    case TIMEOUTS_SET_IMM_IO:
      *pHandleTimeouts = ImmIoTimeouts;
      break;

    default:
      return B_E_PARAM;
    }
  }

  /* note that we're only calling the BestxxxPortTimeoutSet() in case the
   * driver needs its own storage for the timeouts struct. */

  switch (handle_array[handle].port)
  {
    case B_PORT_OFFLINE:          /* use serial as the default */
      if (fSetDefault)
        *pHandleTimeouts = DefaultSerTimeouts;
      break;

    case B_PORT_RS232:
      if (fSetDefault)
        *pHandleTimeouts = DefaultSerTimeouts;
      return BestSerPortTimeoutSet(OsHandle, pHandleTimeouts);

    case B_PORT_PCI_CONF:
    case B_PORT_PCI_IO:
#if 1
      if (fSetDefault)
        *pHandleTimeouts = DefaultPciTimeouts;
      return BestPciPortTimeoutSet(OsHandle, pHandleTimeouts);
#else
      return B_E_OK;
#endif
      
    case B_PORT_FASTHIF:
      if (fSetDefault)
        *pHandleTimeouts = DefaultHifTimeouts;
      return BestHifPortTimeoutSet(OsHandle, pHandleTimeouts);

    default:
      return B_E_WRONG_PORT;
  } /*lint !e788 */
  return B_E_OK;
}


/* --------------------------------------------------------------------------
 * Block mode I/O mechanisms vary between ports...here we take care of
 * those differences and establish common algorithms.
 * NOTE; ALL block mode I/O is streamed !!
 *       Byte-ordering is NOT performed and driver access is 8-bit.
 *       Granularity is the boundary width of an I/O request and can
 *       be any number of bytes.  It ensures that an I/O packet will begin AND
 *       end on the requested boundary.
 *       This could also be thought of as "Sentence Width".
 *       MaxBlockSize is port-dependent.
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * Put the card into block (direct) mode
 * -------------------------------------------------------------------------- */

static b_errtype BestBlockIoSetup(
    b_handletype handle,
    b_int32 offset,             /* starting offset in onboard memory */
    b_int32 num_bytes,          /* length of data at byteptr */
    b_int32 block_bytes,	/* num of bytes per burst */
    b_int8 resource,            /* onboard resource to read/write */
    b_int8 write_cmd            /* CMD_READ or CMD_WRITE */
)
{
  b_errtype err = B_E_OK;
  b_int8 buf[IN_BLOCK_MODE_SET + 1];
  b_int8ptr bp = buf;

  /* as long as we dont' use 'em ... */
  B_USE(block_bytes);

  /* firmware relies on this in block mode set up. CZ */
  assert (CMD_READ == 0x00);  /*lint !e506 */
  assert (CMD_WRITE == 0x01);  /*lint !e506 */
  assert (write_cmd <= CMD_WRITE);
  
  switch (handle_array[handle].port) {
   case B_PORT_FASTHIF:
    /* indication to actually use the fast mode */
    write_cmd |= 0x80;
    break;

   default:
    /* nothing to do */
    break;
  }  /*lint !e788 */
  
  bp = BestLong2Stream(bp, &offset, 1UL);
  bp = BestLong2Stream(bp, &num_bytes, 1UL);

#if 0
  bp = BestLong2Stream(bp, &block_bytes, 1UL);

#endif
  
  bp = BestByteCopy (bp, &resource, 1UL);
  bp = BestByteCopy (bp, &write_cmd, 1UL);
  *bp = 0;

  err = BestBasicCommand(handle, CMD_BLOCK_MODE_SET,
			 buf, IN_BLOCK_MODE_SET,
			 NULL, NULL);

  if (err == B_E_OK)
  {
    BestHwBlockModeBitSet(handle, 1);
  }
  
  return err;
}


/* --------------------------------------------------------------------------
 * Get the card out of block mode
 * -------------------------------------------------------------------------- */

static b_errtype BestBlockIoCleanUp(
    b_handletype handle
)
{
  b_errtype	err = B_E_OK;

  BestHwBlockModeBitSet(handle, 0);

#if 0
  err = BestBasicCommand (handle, CMD_BLOCK_MODE_CLEANUP,
			  NULL, IN_BLOCK_MODE_CLEANUP,
			  NULL, NULL);
#endif
  return err;
}
/* --------------------------------------------------------------------------
 * Block Reads and Writes are identically formed at this level.
 * -------------------------------------------------------------------------- */

static b_errtype BestBlockIo(
    b_handletype handle,
    b_int32 NumBytes,           /* number of bytes */
    int write_cmd,              /* CMD_READ or CMD_WRITE */
    b_int8ptr  pData              /* caller's buffer */
)
{
  B_TRY_VARS_NO_PROG;

  b_int8 RegWidth = BestGetBlockRegWidth(handle);
  
  B_TRY_BEGIN {
    /* the actual I/O ... note that BestBasicRead/Write is block-mode aware */
    if (CMD_READ == write_cmd) {
      B_TRY(BestBasicRead(handle, pData, NumBytes, RegWidth));
    }

    else {
      B_TRY(BestBasicWrite(handle, pData, NumBytes, RegWidth));
    }
  }

  return B_TRY_RET;
}


/* --------------------------------------------------------------------------
 * This is the interface for CAPI programs to perform block (direct) I/O
 * -------------------------------------------------------------------------- */

b_errtype BestBasicBlockCommand(
    b_handletype handle,
    b_blockiorestype resource,    /* onboard resource to read/write */
    b_int32 offset,             /* starting offset in onboard memory */
    b_int32 num_bytes,          /* length of data at byteptr */
    b_int32 granularity,        /* acceptable data element width in bytes */
/* Granularity = 1 means; don't care */
    int write_cmd,              /* CMD_READ or CMD_WRITE */
    b_int8ptr  pData              /* addr of caller's buffer to read/write */
)
{
  B_DECLARE_FUNCNAME ("BestBasicBlockCommand (not exported)");
  B_TRY_VARS;
  b_int32	block_bytes, transfer_bytes;
  b_errtype	err;
  b_porttype 	port;

  /* This call sets up the card for block I/O and flags the handle to
   * indicate that we're in block mode...VERY important ! */

  B_TRY_BEGIN {
    B_TRY (BestGetPortFromHandle(handle, &port));
    
    switch (port) {
     case B_PORT_FASTHIF:
      /* blocks are only needed for FAST HIF */
      block_bytes = 0x1eff0L;
      B_TRY_FCT_PARAM_ALIGNMENT (block_bytes, granularity);
      break;
      
     default:
#if 0
      block_bytes = best_capi_prop [handle] [B_CAPI_BLOCKMAX];
      block_bytes -= block_bytes % granularity;

#else
      block_bytes = num_bytes;

#endif
      
      break;
    }  /*lint !e788 */
    
    B_TRY_FAIL (block_bytes == 0 ? B_E_PARAM : B_E_OK);
    B_TRY_FCT_PARAM_ALIGNMENT (granularity, 2L);
    B_TRY_FCT_PARAM_ALIGNMENT (offset, granularity);
    B_TRY_FCT_PARAM_ALIGNMENT (num_bytes, granularity);
    
    B_TRY_PROGRESS (BestBlockIoSetup (handle, offset, num_bytes, block_bytes,
				      (b_int8) resource, (b_int8) write_cmd));
    DBG_CB_PRINTF (("BBBC: setup\n"));
    
    while (num_bytes) {
      transfer_bytes = num_bytes < block_bytes ? num_bytes : block_bytes;
    
      /* Do the actual I/O. Note that we must NEVER return without calling
       * BestBlockIoCleanUp() ... that clears the handle's block I/O flag. */
      B_TRY (BestBlockIo(handle, transfer_bytes, write_cmd, pData));
      DBG_CB_PRINTF (("BBBC: block IO\n"));

      num_bytes -= transfer_bytes;
      pData += transfer_bytes;
    }

    B_TRY_RESET;
    B_TRY (BestBlockIoCleanUp (handle));
    DBG_CB_PRINTF (("BBBC: clean up\n"));
  }
  
  B_TRY_CATCH {
    B_TRY_PASSED {
      err = BestBlockIoCleanUp(handle);

      if (B_TRY_RET == B_E_OK)
	B_TRY_RET = err;
    }
  }

  return B_TRY_RET;
}


/* -------------------------------------------------------------------- */
/* control functions */

void BestCBReadStringSet (b_handletype handle, char *s)
{
  if (handle < MAXHANDLES)
    cb_read_string_array [handle] = s;
}


void BestCBWriteStringSet (b_handletype handle, char *s)
{
  if (handle < MAXHANDLES)
    cb_write_string_array [handle] = s;
}


